const opcua = require("node-opcua");
const sendTriggers = require("../../utils/sendTriggers");
const logger = require("../../utils/logger");
const clients = require("../clients");

module.exports = function opcuaSubscribe(message, ws) {
  const { name, variableName, trigger, entity } = message;
  const client = clients[name];

  if (!client || !client.session) {
    const error = `Client '${name}' not connected`;
    logger.error("Error " + error);
    return sendTriggers(ws, "BRIDGE_ERROR", trigger, entity, { error });
  }

  const entry = client.map?.[variableName];
  const nodeId = entry?.nodeId;

  if (!nodeId) {
    const error = `Variable '${variableName}' not found in map`;
    logger.warn(error);
    return sendTriggers(ws, "BRIDGE_ERROR", trigger, entity, { error });
  }

  const subscription = opcua.ClientSubscription.create(client.session, {
    requestedPublishingInterval: 250,
    requestedLifetimeCount: 100,
    requestedMaxKeepAliveCount: 10,
    maxNotificationsPerPublish: 50,
    publishingEnabled: true,
    priority: 10,
  });

  const monitor = opcua.ClientMonitoredItem.create(
    subscription,
    {
      nodeId,
      attributeId: opcua.AttributeIds.Value,
    },
    { samplingInterval: 250, discardOldest: true, queueSize: 10 },
    opcua.TimestampsToReturn.Both
  );

  monitor.on("changed", (dataValue) => {
    let value = dataValue.value?.value;
    if (typeof value === "boolean") value = value ? 1 : 0;

    const payload = {
      VALUE: value,
      VARNAME: variableName,
      TIMESTAMP: Date.now(),
    };

    sendTriggers(ws, "BRIDGE_SUBSCRIBE_RESULT", trigger, entity, {
      log: "",
      map: payload,
    });
  });

  if (!client.subscriptions.some(s => s.variableName === variableName)) {
    client.subscriptions.push({
      variableName,
      subscription,
      monitor,
      nodeId,
      trigger,
      entity,
    });
  }

  logger.info(
    `✅ Subscribed to '${variableName}' (${nodeId}) for client '${client.name}'`
  );
};
