const { AttributeIds } = require("node-opcua");
const clients = require("../clients");
const sendTriggers = require("../../utils/sendTriggers");
const logger = require("../../utils/logger");

module.exports = async function opcuaRead(message, ws) {
  const { name, variables, trigger, entity } = message;
  const client = clients[name];

  if (!client || !client.session) {
    const error = `Client '${name}' not connected`;
    logger.error("Error " + error);
    return sendTriggers(ws, "BRIDGE_ERROR", trigger, entity, { error });
  }

  try {
    const map = client.map || {};
    const varNames = Array.isArray(variables)
      ? variables
      : Object.keys(variables || {});

    const nodesToRead = [];

    for (const varName of varNames) {
      const entry = map[varName];
      const nodeId = entry?.nodeId;

      if (!nodeId) {
        return sendTriggers(ws, "BRIDGE_ERROR", trigger, entity, {
          error: `Variable '${varName}' not found in map`,
        });
      }

      nodesToRead.push({ nodeId, attributeId: AttributeIds.Value });
    }

    const dataValues = await client.session.read(nodesToRead);

    const result = {};


varNames.forEach((name, idx) => {
  const dv = dataValues[idx];
  const sc = dv?.statusCode?.toString() ?? "NoStatus";
  const val = dv?.value?.value;

  if (!dv) {
    console.warn(`No dataValue for ${name}`);
    result[name] = null;
    return;
  }

  if (!dv.value || val === undefined) {
    console.warn(`Null or undefined value for ${name}, statusCode=${sc}`);
    result[name] = null;
    return;
  }

  let value = val;
  if (typeof value === "boolean") value = value ? 1 : 0;


if (typeof value === "string") {
    // Try parse el JSON
    try {
        const json_value = JSON.parse(value);
        // Si el parseo es exitoso Y es un objeto/array, usamos el objeto.
        if (typeof json_value === 'object' && json_value !== null) {
            value = json_value;
            console.log(`[BRIDGE-READ] Parsed JSON for ${name}`);
        }
    } catch (e) {
        // Ignorar errores de parseo (la cadena era solo un string normal)
    }
}




  result[name] = value;

  console.log(`Read ${name}:`, {
    nodeId: nodesToRead[idx].nodeId.toString(),
    statusCode: sc,
    value
  });
});


    return sendTriggers(ws, "BRIDGE_READ_RESULT", trigger, entity, {
      log: "",
      map: result,
    });
  } catch (err) {
    const error = `OPCUA read error: ${err.message}`;
    logger.error("Error " + error);
    return sendTriggers(ws, "BRIDGE_ERROR", trigger, entity, { error });
  }
};
