const opcua = require("node-opcua");
const sendTriggers = require("../../utils/sendTriggers");
const logger = require("../../utils/logger");
const clients = require("../clients");
const opcuaSubscribe = require("./opcuaSubscribe");

async function opcuaInit(endpoint, options = {}, name, ws) {
  try {
    const client = opcua.OPCUAClient.create({
      endpointMustExist: false,
      connectionStrategy: { initialDelay: 1000, maxRetry: 1 },
      ...options.opcuaConfig,
    });

    // Store credentials if provided
    const credentials = options.user && options.pass
      ? { userName: options.user, password: options.pass }
      : undefined;

    // EVENT HANDLERS
    client.on("close", () => {
      logger.warn(`OPC-UA client '${name}' connection closed`);
		
		cleanupClient(name);		
		
      if (clients[name]) clients[name].connected = false;

      sendTriggers(ws, "BRIDGE_ERROR", "", null, {
        error: `OPC-UA connection for '${name}' closed`,
        name,
      });
    });

    client.on("backoff", (attempt, delay) => {
      logger.warn(`Reconnecting '${name}' in ${delay}ms (attempt ${attempt})`);
      sendTriggers(ws, "BRIDGE_ERROR", "", null, {
        error: `Reconnecting '${name}' in ${delay}ms (attempt ${attempt})`,
        name,
      });
    });

    async function restoreSession(name, sessionOverride = null) {
      try {
        const oldSession = clients[name]?.session;
        if (oldSession && oldSession !== sessionOverride) {
          try {
            await oldSession.close();
          } catch (err) {
            logger.warn(`Could not close old session: ${err.message}`);
          }
        }

        const creds = clients[name]?.credentials;
        const session = sessionOverride || await client.createSession(creds);
        clients[name].session = session;

        const oldSubs = [...(clients[name].subscriptions || [])];
        clients[name].subscriptions = [];

        for (const sub of oldSubs) {
          opcuaSubscribe({
            name,
            variableName: sub.variableName,
            trigger: sub.trigger,
            entity: sub.entity
          }, clients[name].ws);
        }

        logger.info(`Subscriptions restored for '${name}'`);
      } catch (err) {
        logger.error(`Error restoring session for '${name}': ${err.message}`);
      }
    }

    client.on("after_reconnection", async () => {
      logger.info(`Reconnection completed for '${name}'`);
      await restoreSession(name);
    });

    client.on("connection_reestablished", async () => {
      logger.warn(`OPC-UA connection reestablished for ${name}`);
      await restoreSession(name);
    });

    client.on("new_session_created", async (session) => {
      logger.info(`New OPC-UA session created for '${name}'`);
      await restoreSession(name, session);
    });

    client.on("connection_lost", () => {
      logger.error(`Connection lost for OPC-UA client '${name}'`);
    });

    await client.connect(endpoint);
    const session = await client.createSession(credentials);

    return {
      client,
      session,
      credentials // stored for reuse in reconnections
    };

  } catch (err) {
    return { error: `OPCUA init failed: ${err.message}` };
  }
}


function cleanupClient(name) {
  const client = clients[name];
  if (!client) return;

  // Cerrar subscripciones
  if (client.subscriptions?.length) {
    for (const sub of client.subscriptions) {
      sub.subscription?.terminate?.().catch(e =>
        logger.warn(`Error terminando subscripción '${sub.variableName}': ${e.message}`)
      );
    }
  }

  // Cerrar sesión
  client.session?.close?.().catch(() => {});
  client.client?.disconnect?.().catch(() => {});

  delete clients[name];
  logger.info(`Client '${name}' limpiado tras desconexión remota`);
}



module.exports = opcuaInit;
