// src/services/opcuaCall.js

const clients = require("../clients");
const sendTriggers = require("../../utils/sendTriggers");
const logger = require("../../utils/logger");
const { StatusCodes, DataType } = require("node-opcua");

// Conversión genérica de JS → OPC-UA Variant
function toVariant(val) {
  if (val === null || val === undefined) {
    return { dataType: DataType.Null, value: null };
  }

  const t = typeof val;

  if (t === "number") {
    return Number.isInteger(val)
      ? { dataType: DataType.Int32, value: val }
      : { dataType: DataType.Double, value: val };
  }

  if (t === "string") {
    return { dataType: DataType.String, value: val };
  }

  if (t === "boolean") {
    return { dataType: DataType.Boolean, value: val };
  }

  if (val instanceof Date && !isNaN(val.getTime())) {
    return { dataType: DataType.DateTime, value: val };
  }

  if (Buffer.isBuffer(val) || val instanceof Uint8Array) {
    return { dataType: DataType.ByteString, value: Buffer.from(val) };
  }

  // Objetos y arrays → se serializan a JSON string
  try {
    return { dataType: DataType.String, value: JSON.stringify(val) };
  } catch {
    return { dataType: DataType.String, value: String(val) };
  }
}

module.exports = async function opcuaCall(message, ws) {
  const { name, process, params, trigger, entity } = message;
  const client = clients[name];

  if (!client || !client.session) {
    const error = `Client '${name}' not connected`;
    logger.error(error);
    return sendTriggers(ws, "BRIDGE_ERROR", trigger, entity, { error });
  }

  // Los parámetros deben ser un array
  if (!Array.isArray(params)) {
    const error = `BRIDGE_CALL 'params' must be an array`;
    logger.error(error);
    return sendTriggers(ws, "BRIDGE_ERROR", trigger, entity, { error });
  }

  const map = client.map || {};
  const entry = map[process];

  if (!entry || entry.dataType !== "Method") {
    const error = `Process '${process}' not found or not a Method in map`;
    logger.error(error);
    return sendTriggers(ws, "BRIDGE_ERROR", trigger, entity, { error });
  }

  // Obtener el objeto padre real desde el browse
  const objectId = entry.parentNodeId;
  if (!objectId) {
    const error = `Parent object not found for method '${process}'`;
    logger.error(error);
    return sendTriggers(ws, "BRIDGE_ERROR", trigger, entity, { error });
  }

  try {
    // Convertir cada elemento del array en un Variant
    const inputArguments = params.map(toVariant);

    const callRequest = {
      objectId,
      methodId: entry.nodeId,
      inputArguments,
    };

    logger.info(`[OPCUA CALL] ${process} → ${JSON.stringify(inputArguments.map(a => a.value))}`);

    const callResult = await client.session.call(callRequest);

	if (callResult.statusCode !== StatusCodes.Good) {
	  let detail = "";
	  try {
		const errVal = callResult.outputArguments?.[0]?.value;
		if (typeof errVal === "string") detail = errVal;
		else if (errVal && typeof errVal === "object") detail = JSON.stringify(errVal);
	  } catch {}
	  const error = `OPC-UA call failed (${callResult.statusCode.toString()}): ${detail}`;
	  logger.error(error);
	  return sendTriggers(ws, "BRIDGE_ERROR", trigger, entity, { error });
	}


    // Leer la salida (primer argumento de salida)
    let output = callResult.outputArguments?.[0]?.value ?? null;
    if (typeof output === "string") {
      try { output = JSON.parse(output); } catch { /* mantener string */ }
    }

    logger.info(`[OPCUA CALL] ${process} OK → ${JSON.stringify(output)}`);

    return sendTriggers(ws, "BRIDGE_CALL_RESULT", trigger, entity, {
      log: `[${process}] executed successfully.`,
      map: output || {},
    });


  } catch (err) {
    const error = `OPCUA call error: ${err.message}`;
    logger.error(error);
    return sendTriggers(ws, "BRIDGE_ERROR", trigger, entity, { error });
  }
};

