const clients = require("../clients");
const sendTriggers = require("../../utils/sendTriggers");
const logger = require("../../utils/logger");
const { AttributeIds, DataTypeIds, NodeClass } = require("node-opcua");

async function recursiveBrowse(session, nodeId, path = "", depth = 0, result = []) {
  let browseResult;
		try {
		  browseResult = await session.browse({
			nodeId,
			referenceTypeId: null,          // incluir todas las referencias
			includeSubtypes: true,
			browseDirection: 0,             // hacia hijos
			nodeClassMask: 0,               // sin filtro
			resultMask: 63,                 // obtener todo lo posible
		  });
		} catch (err) {
		  logger.error(`[BROWSE] Error navegando ${nodeId}: ${err.message}`);
		  return result;
		}

		if (!browseResult.references || browseResult.references.length === 0) {
		  // Sin referencias, retorna vaco pero no rompas la recursin
		  return result;
		}

  for (const ref of browseResult.references) {
    const nodeIdStr = ref.nodeId.toString();
    const namespaceIndex = ref.nodeId.namespace;
    const fullPath = path ? `${path}_${ref.browseName.name}` : ref.browseName.name;

    if (namespaceIndex > 0) {
      let writable = 0;
      let dataType = "Unknown";

      if (ref.nodeClass === NodeClass.Variable || ref.nodeClass === NodeClass.Method) {
        if (ref.nodeClass === NodeClass.Variable) {
          try {
            const [accessLevelAttr, dataTypeAttr] = await session.read([
              { nodeId: ref.nodeId, attributeId: AttributeIds.UserAccessLevel },
              { nodeId: ref.nodeId, attributeId: AttributeIds.DataType },
            ]);

            const val = accessLevelAttr?.value?.value ?? 0;
            writable = (val & 0x02) !== 0 ? 1 : 0;

            const dtNodeId = dataTypeAttr?.value?.value;
            if (dtNodeId?.namespace === 0 && typeof dtNodeId.value === "number") {
              dataType =
                Object.entries(DataTypeIds).find(([, val]) => val === dtNodeId.value)?.[0] ||
                `i=${dtNodeId.value}`;
            } else {
              dataType = dtNodeId?.toString() || "Unknown";
            }
          } catch (err) {
            logger.warn(`No attrs from ${nodeIdStr}: ${err.message}`);
          }
        } else if (ref.nodeClass === NodeClass.Method) {
          dataType = "Method";
        }

        // 👇 Aquí añadimos parentNodeId
        result.push({
          browseName: ref.browseName.name,
          nodeId: nodeIdStr,
          parentNodeId: nodeId, // <--- el nodo que contiene este método/variable
          path: fullPath,
          writable,
          dataType,
        });
      }
    }

	if (
	  ref.nodeClass !== NodeClass.Method &&
	  depth < 3 &&
	  ref.nodeId.namespace > 0
	) {
	  await recursiveBrowse(session, ref.nodeId, fullPath, depth + 1, result);
	}
	  }

  return result;
}


function processBrowseResult(rawNodes) {
  const map = {};
  for (const item of rawNodes) {
    map[item.path] = {
      nodeId: item.nodeId,
      parentNodeId: item.parentNodeId || null,
      writable: item.writable === 1,
      dataType: item.dataType,
    };
  }

  const log = rawNodes.map(item => {
    const parts = [`${item.path}: ${item.nodeId}`];
    if (item.dataType) {
      parts.push(item.dataType === "Method" ? "[Method]" : `[${item.dataType}]`);
    }
    if (item.writable) parts.push(`(writable)`);
    return parts.join(" ");
  }).join("\n");

  return { map, log };
}

module.exports = async function opcuaBrowse(message, ws) {
  const { name, trigger, entity } = message;
  const client = clients[name];

  if (!client || !client.session) {
    const error = `Client '${name}' not connected`;
    logger.error(error);
    return sendTriggers(ws, "BRIDGE_ERROR", trigger, entity, { error });
  }

  try {
    const raw = await recursiveBrowse(client.session, "ns=0;i=85");
    const { map, log } = processBrowseResult(raw);
    client.map = map;

    return sendTriggers(ws, "BRIDGE_BROWSE_RESULT", trigger, entity, { log, map });
  } catch (err) {
    logger.error(`Browse error: ${err.message}`);
    return sendTriggers(ws, "BRIDGE_ERROR", trigger, entity, { error: err.message });
  }
};
