// src/services/mqtt/mqttWrite.js

const clients = require("../clients");
const sendTriggers = require("../../utils/sendTriggers");
const logger = require("../../utils/logger");

/**
 * Procesa un comando BRIDGE_WRITE para publicar mensajes MQTT.
 * Esta versión está corregida para usar directamente el tópico y payload del mensaje de entrada.
 *
 * @param {object} message - El mensaje entrante de GPSS-Plus.
 * @param {object} ws - El objeto de conexión WebSocket.
 */
module.exports = async function mqttWrite(message, ws) {
  const { name, writes, trigger, entity } = message;
  const client = clients[name];

  // La conexión es válida si 'client' y su propiedad 'client' existen.
  // Se ha corregido la referencia para que coincida con lo que devuelve mqttInit.js.
  if (!client || !client.client) {
    const error = `MQTT client '${name}' not connected or not initialized`;
    logger.error(`[MQTT:${name}] ${error}`);
    // Enviar un mensaje de error al cliente si el puente no está listo
    return sendTriggers(ws, "BRIDGE_ERROR", trigger, entity, { error });
  }

  const results = {};

  // Iterar directamente sobre las claves y valores del objeto 'writes'.
  // En tu caso, la clave (topic) es "devices/elprimero/commands/led"
  // y el valor (payload) es "on" o "off".
  for (const [topic, payload] of Object.entries(writes || {})) {
    try {
      // Publicar directamente el payload. NO se anida en un JSON.
      // Se ha corregido la referencia a client.client para la publicación.
      client.client.publish(topic, payload);
      logger.info(`[MQTT:${name}] → Published to ${topic}: ${payload}`);
      results[topic] = "OK";
    } catch (err) {
      results[topic] = `Error: ${err.message}`;
      logger.error(`[MQTT:${name}] Publish error: ${err.message}`);
    }
  }

  // Enviar el resultado de la operación
  sendTriggers(ws, "BRIDGE_WRITE_RESULT", trigger, entity, {
    log: "",
    map: results,
  });
};
