// src/services/mqtt/mqttSubscribe.js

const sendTriggers = require("../../utils/sendTriggers");
const logger = require("../../utils/logger");
const clients = require("../clients");

module.exports = function mqttSubscribe(message, ws) {
  const { name, variableName, trigger, entity } = message;
  const client = clients[name];

  if (!client || !client.client || !client.map) {
    const error = `MQTT client '${name}' not connected or not initialized`;
    logger.error(`[MQTT:${name}] ${error}`);
    return sendTriggers(ws, "BRIDGE_ERROR", trigger, entity, { error });
  }

  const mqtt = client.client;
  const topic = variableName;

  // Evita múltiples suscripciones
  if (!client.subscriptions.some(s => s.variableName === topic)) {
    client.subscriptions.push({ variableName: topic, trigger, entity });

    mqtt.subscribe(topic, (err) => {
      if (err) {
        logger.error(`[MQTT:${name}] ❌ Error al suscribirse a ${topic}: ${err.message}`);
      } else {
        logger.info(`[MQTT:${name}] 🟢 Suscrito a ${topic}`);
      }
    });

    // 📤 Enviamos comando al ESP vía DEVICE_ID/cmd
    const pathParts = topic.split("/");
    const deviceId = pathParts[0];
//	  console.log("DEBUG client object:", clients[name]);

    if (client.publishCommand) {
      const cmd = `SUBSCRIBE ${topic}`;
      client.publishCommand(deviceId, cmd);
    } else {
      logger.warn(`[MQTT:${name}] ⚠️ No publishCommand method available`);
    }
  }

  // Listener único global
  if (!client._mqttOnMessage) {
    client._mqttOnMessage = (receivedTopic, messageBuffer) => {
      const subscription = client.subscriptions.find(s => s.variableName === receivedTopic);
      if (!subscription) return;

      try {
        const payload = JSON.parse(messageBuffer.toString());
        const key = Object.keys(payload)[0]; // p.ej. 'lux', 'value', etc.
        const value = payload[key];

        const result = {
          VALUE: value,
          VARNAME: receivedTopic,
          TIMESTAMP: Date.now()
        };

        sendTriggers(ws, "BRIDGE_SUBSCRIBE_RESULT", subscription.trigger, subscription.entity, {
          log: "",
          map: result
        });

      } catch (err) {
        logger.warn(`[MQTT:${name}] ⚠️ Error parsing message on ${receivedTopic}: ${err.message}`);
      }
    };

    mqtt.on("message", client._mqttOnMessage);
  }
};
