// src/services/mqtt/mqttInit.js

const mqtt = require("mqtt");
const sendTriggers = require("../../utils/sendTriggers");
const logger = require("../../utils/logger");

async function mqttInit(endpoint, options = {}, name, ws) {
  try {
    const brokerUrl = endpoint.replace("mqtt_embedded://", "mqtt://localhost");

    const clientId = `bridge_${name}_${Math.floor(Math.random() * 10000)}`;
    const client = mqtt.connect(brokerUrl, {
      clientId,
      keepalive: 60,
      reconnectPeriod: 5000,
      connectTimeout: 5000,
    });

    // Función auxiliar para enviar comandos al topic `deviceId/cmd`
    function publishCommand(deviceId, command) {
      const topic = `${deviceId}/cmd`;
      client.publish(topic, command, (err) => {
        if (err) {
          logger.error(`[MQTT:${name}] ❌ Error publishing to ${topic}: ${err.message}`);
        } else {
          logger.info(`[MQTT:${name}] 📤 Published command to ${topic}: ${command}`);
        }
      });
    }

    client.on("connect", () => {
      logger.info(`[MQTT:${name}] ✅ Connected to ${brokerUrl}`);
    });

    client.on("error", (err) => {
      logger.error(`[MQTT:${name}] ❌ Error: ${err.message}`);
      sendTriggers(ws, "BRIDGE_ERROR", "", null, {
        error: err.message,
        name,
      });
    });

    client.on("close", () => {
      logger.warn(`[MQTT:${name}] 🔌 Disconnected`);
      sendTriggers(ws, "BRIDGE_ERROR", "", null, {
        error: "MQTT connection closed",
        name,
      });
    });

    // Devuelve todo lo necesario para clients[name]
    return {
      client,
      publishCommand,
      announces: {} // (por si quieres cachear announces a futuro)
    };

  } catch (err) {
    return { error: `MQTT init failed: ${err.message}` };
  }
}

module.exports = mqttInit;
