const clients = require("../clients");
const sendTriggers = require("../../utils/sendTriggers");
const logger = require("../../utils/logger");

module.exports = async function mqttBrowse(message, ws) {
  const { name, trigger, entity } = message;
  const client = clients[name];

  if (!client || !client.client) {
    const error = `Client '${name}' not connected`;
    logger.error(`[MQTT:${name}] ${error}`);
    return sendTriggers(ws, "BRIDGE_ERROR", trigger, entity, { error });
  }

  const mqtt = client.client;
  const announces = {}; // Aquí guardamos todos los announces recibidos
  const timeoutMs = 1000; // Tiempo de escucha de announces

  logger.info(`[MQTT:${name}] 🔍 Escuchando announces durante ${timeoutMs}ms...`);

  const onMessage = (topic, payload) => {
    topic = topic.trim();
    const match = topic.match(/^(.+)\/announce$/);
    if (!match) return;

    const deviceId = match[1];

    try {
      const data = JSON.parse(payload.toString());
      const vars = data.variables || [];

      announces[deviceId] = {};
      for (const v of vars) {
        announces[deviceId][v.path] = {
          nodeId: v.path,
          writable: !!v.writable,
          dataType: v.type || "unknown"
        };
      }

    } catch (err) {
      logger.warn(`[MQTT:${name}] ❌ Error parsing announce from ${deviceId}: ${err.message}`);
    }
  };

  // Suscribirse temporalmente a announces
  mqtt.subscribe("+/announce", (err) => {
    if (err) {
      const error = `Failed to subscribe to announce topics: ${err.message}`;
      logger.error(`[MQTT:${name}] ${error}`);
      return sendTriggers(ws, "BRIDGE_ERROR", trigger, entity, { error });
    }

    mqtt.on("message", onMessage);

    setTimeout(() => {
      mqtt.removeListener("message", onMessage);

      logger.info(`[MQTT:${name}] 📡 Announces recopilados: ${Object.keys(announces).length}`);

      // Generar log y map como espera el frontend
      const map = {};
      const logLines = [];

      for (const deviceId in announces) {
        for (const path in announces[deviceId]) {
          const fullPath = `${deviceId}${path}`;
          const v = announces[deviceId][path];
          map[fullPath] = v;
          logLines.push(`${fullPath} [${v.dataType}]${v.writable ? " (writable)" : ""}`);
        }
      }

      client.map = map; // Guardamos el map en memoria

      sendTriggers(ws, "BRIDGE_BROWSE_RESULT", trigger, entity, {
        map,
        log: logLines.join("\n")
      });

    }, timeoutMs);
  });
};
