// src/handlers/handleBridgeInit.js

const clients = require("../services/clients");
const sendTriggers = require("../utils/sendTriggers");


const opcuaInit = require("../services/opcua/opcuaInit");
const mqttInit = require("../services/mqtt/mqttInit");




function inferTypeFromClient(client) {
  if (client.startsWith("opc.tcp://")) return "OPCUA";
  if (client.startsWith("mqtt://")) return "MQTT";
  if (client.startsWith("mqtt_embedded://")) return "MQTT";
	
  return null;
}

async function handleBridgeInit(message, ws) {
  const { client, options = {}, name } = message;
  const type = inferTypeFromClient(client);

  if (!type) {
    return sendTriggers(ws, "BRIDGE_ERROR", "", null, {
      error: "BRIDGER type not recognized"
    });
  }

  try {
    let initResult;

    switch (type) {
      case "OPCUA":
        initResult = await opcuaInit(client, options, name, ws);
        break;
      case "FS":
        initResult = await fsInit(client, options, name, ws);
        break;
      case "MYSQL":
        initResult = await mysqlInit(client, options, name, ws);
        break;
      case "SERIAL":
        initResult = await serialInit(client, options, name, ws);
        break;
      case "MQTT":
        initResult = await mqttInit(client, options, name, ws);
        break;
    }

    if (initResult.error) {
      console.error(`[ERROR] Init failed for '${name}': ${initResult.error}`);
      return sendTriggers(ws, "BRIDGE_ERROR", "", null, {
        error: initResult.error
      });
    }

    // Guardar solo lo necesario
    clients[name] = {
      name,
      type,
      ws,
      ...initResult,
      subscriptions: []
    };

    return sendTriggers(ws, "BRIDGE_CONNECT_OK", "", null, {
      log: "BRIDGE_CONNECT_OK"
    });

  } catch (err) {
    console.error(`[ERROR] Init crash for '${name}': ${err.message}`);
    return sendTriggers(ws, "BRIDGE_ERROR", "", null, {
      error: `Connection error: ${err.message}`
    });
  }
}

module.exports = { handleBridgeInit };
