const { DataType } = require("node-opcua");
const expose_ro = require("../utils/expose_ro");
const expose_rw = require("../utils/expose_rw");

let currentWeight = 0.0;
let targetWeight = 0.0;
let isStabilizing = false;
let stabilizationStart = 0;
let forceStable = false;

let triggerValue = false;

function register(namespace) {
    const weightNode = namespace.addVariable({
        organizedBy: namespace.addressSpace.rootFolder.objects,
        browseName: "Device_Scale_Weight",
        dataType: DataType.Double
    });
    expose_ro(weightNode, () => currentWeight);

    const triggerNode = namespace.addVariable({
        organizedBy: namespace.addressSpace.rootFolder.objects,
        browseName: "Device_Scale_Trigger",
        dataType: DataType.Boolean
    });

    expose_rw(
        triggerNode,
        () => triggerValue,
        (val) => {
            triggerValue = Boolean(val);
            isStabilizing = true;
            forceStable = false;
            stabilizationStart = Date.now();

            targetWeight = triggerValue
                ? Math.round((Math.random() * 10 + 5) * 100) / 100  // 5.00 - 15.00
                : 0.0;
        }
    );
}

function update() {
    if (!isStabilizing) return;

    const now = Date.now();
    const elapsed = now - stabilizationStart;
    const progress = elapsed / 2000;

    if (elapsed >= 2000) {
        currentWeight = targetWeight;
        isStabilizing = false;
        return;
    }

    // En los últimos 400ms, forzamos estabilización final cerca del valor final
    if (elapsed >= 1600 && !forceStable) {
        currentWeight = targetWeight + (Math.random() * 0.2 - 0.1);  // ±0.1
        forceStable = true;
        return;
    }

    // Simulación suave progresiva
    const step = (targetWeight - currentWeight) * progress * 0.2;
    currentWeight += step;

    // Redondeo a 3 decimales
    currentWeight = Math.round(currentWeight * 1000) / 1000;
}

module.exports = { register, update };
