const {
    OPCUAServer,
    SecurityPolicy,
    MessageSecurityMode,
    UserTokenType,
    UserTokenPolicy
} = require("node-opcua");

const args = require("minimist")(process.argv.slice(2));
const port = args.port || 4840;

const sensorMotion = require("./sensors/sensor_motion");
const sensorDoor = require("./sensors/sensor_door");
const sensorTemperature = require("./sensors/sensor_temperature");
const sensorMotor = require("./sensors/device_motor");
const sensorBattery = require("./sensors/sensor_battery");
const sensorScaleControlled = require("./sensors/sensor_scale_controlled");
const sensorPiston = require("./sensors/sensor_pistons");

const server = new OPCUAServer({
    port: port,
    hostname: "opcua.gpss-plus.com",
    resourcePath: "",
    allowAnonymous: false,
    securityPolicies: [SecurityPolicy.None],
    securityModes: [MessageSecurityMode.None],
    buildInfo: {
        productName: "VirtualPlant",
        buildNumber: "1",
        buildDate: new Date()
    },
    userManager: {
        isValidUser: (username, password) => {
            const users = {
                "user2": "1234",
                "user": "1234"
            };
            return users[username] === password;
        }
    }
});

let intervalId = null;
let clientCount = 0;

async function startSensorLoop() {
    if (!intervalId) {
        console.log("Starting...");
        intervalId = setInterval(() => {
            sensorMotion.update();
            sensorDoor.update();
            sensorTemperature.update();
            sensorMotor.update();
            sensorBattery.update();
            sensorScaleControlled.update();
            sensorPiston.update();
        }, 100);
    }
}

function stopSensorLoop() {
    if (intervalId) {
        clearInterval(intervalId);
        intervalId = null;
        console.log("Stop...");
    }
}

(async () => {
    await server.initialize();

    const addressSpace = server.engine.addressSpace;
    const namespaceIndex = addressSpace.registerNamespace("http://opcua.gpss-plus.com");
    const namespace = addressSpace.getOwnNamespace();

    // Registrar sensores
    sensorMotion.register(namespace);
    sensorDoor.register(namespace);
    sensorTemperature.register(namespace);
    sensorMotor.register(namespace);
    sensorBattery.register(namespace);
    sensorScaleControlled.register(namespace);
    sensorPiston.register(namespace);

    // Configurar políticas de autenticación
    const endpoint = server.endpoints[0];
    endpoint.endpointDescriptions().forEach(desc => {
        desc.userIdentityTokens = [
            new UserTokenPolicy({
                policyId: "username",
                tokenType: UserTokenType.UserName,
                issuedTokenType: null,
                issuerEndpointUrl: null,
                securityPolicyUri: SecurityPolicy.None
            })
        ];
    });

    await server.start();
    console.log("OPC-UA running: ", endpoint.endpointDescriptions()[0].endpointUrl);

    // Eventos de conexión / desconexión
    server.on("session_activated", () => {
        clientCount++;
        console.log(`Client connecting. Total: ${clientCount}`);
        if (clientCount === 1) {
            startSensorLoop();
        }
    });

    server.on("session_closed", () => {
        clientCount = Math.max(0, clientCount - 1);
        console.log(`Client disconnecting. Total: ${clientCount}`);
        if (clientCount === 0) {
            stopSensorLoop();
        }
    });
})();
